function VC = first_order(varargin)

% VC = first_order(A1,A2,A4,SIGMA,ns) 
% Solves a first-order approximation of a stochastic dynamic model 
% and returns the time-paths of the second moments.  
%
% The following input is required:
% 1. 	The coefficient matrices of the first-order system - A1,A2,A4.
% 2.    The variance-covariance matrix of the iid shocks - SIGMA. 
% 3.	The total number of (endogenous and exogenous) predetermined 
%       variables - ns.
%
% Two further optional arguments may be provided: hr1 is the number of 
% time periods for which solutions for first moments will be generated 
% and hr2 is the number of additional time periods for which conditional 
% second moments are required.  By default hr1=hr2=200 so first_order 
% generates conditional second moments for 400 periods.
%
% Some sections of code relating to the QZ decomposition are taken
% from solab.m written by Paul Klein.
%
% Calls qzdiv() (by Christopher Sims)
%
% Written by Alan Sutherland, 24 September 2003
%

A1=varargin{1};
A2=varargin{2};
A4=varargin{3};
SIGMA=varargin{4};
ns=varargin{5};

if length(varargin)==7
    hr1=varargin{6};
    hr2=varargin{7};
else
    hr1=200;
    hr2=200;
end    

[nd,nd]=size(A1);
nu=nd-ns;

if length(varargin)==8
    SV=varargin{8};
else
    SV=zeros(nd,nd);
end

hr3=hr1+hr2;

stake=1.000001;

[s,t,qk,z] = qz(A1,A2);            
[s,t,qk,z] = qzdiv(stake,s,t,qk,z); 

z11=z(1:ns,1:ns);
z21=z(ns+1:end,1:ns);

if rank(z11)<ns;
 error('Invertibility condition violated')
end

if abs(t(ns,ns))>stake*abs(s(ns,ns)) | abs(t(ns+1,ns+1))<stake*abs(s(ns+1,ns+1));
 warning('Wrong number of stable eigenvalues.');
end

z11i=z11\eye(ns);

s11=s(1:ns,1:ns);
t11=t(1:ns,1:ns);

dyn=s11\t11;

G=real(z11*dyn*z11i);
F=real(z21*z11i); 

D=A4(1:ns,:);

H=[G zeros(ns,nu); F*G zeros(nu,nu)];
C=[D; F*D];

% generate conditional second moments

VC=zeros(nd,nd,hr3+4);

VC(:,:,1)=SV;

for tt=2:hr3+4
    VC(:,:,tt)=H*VC(:,:,tt-1)*H'+C*SIGMA*C';
end