#pragma once

//* MS openGL libraries (link with OPENGL32.LIB and GLU32.LIB)
#include "gl\gl.h"
#include "gl\glu.h"

/*
Using GDI with OpenGL
Lots of documents say that if you use double-buffering, you can't
use GDI calls because it doesn't know about back buffers. 
Some sources say that you can if you dump the OpenGL rendering 
context. This certainly works on some computers (mine), 
but may cause crashes on others. MSDN says that "Once the 
pixel format for a window has been set (by calling SetPixelFormat 
with a DC of that window), it can never be reset", which rather 
suggests that you can't mix them.

If USE_GDI = TRUE, then program allows GDI calls.
If USE_GDI = FALSE, then program does not compile GDI call stuff.
*/
#define USE_GDI 0

//#define GLdouble double
/////////////////////////////////////////////////////////////////////////////
// Global type definitions
	enum InfoField {VENDOR,RENDERER,VERSION,ACCELERATION,EXTENSIONS};
	enum ColorsNumber{INDEXED,THOUSANDS,MILLIONS,MILLIONS_WITH_TRANSPARENCY};
	enum ZAccuracy{NORMAL,ACCURATE};

/////////////////////////////////////////////////////////////////////////////
// general 3D classes

class CPoint3D
{
public:
   float x, y, z;
   CPoint3D () { x=y=z=0; }
   CPoint3D (float c1, float c2, float c3)
   {
      x = c1;      y = c2;      z = c3;
   }
   CPoint3D& operator=(const CPoint3D& pt)
   {
      x = pt.x;   z = pt.z;   y = pt.y;
      return *this;
   }
   CPoint3D (const CPoint3D& pt)
   {
      *this = pt;
   }
   void Translate(float cx, float cy, float cz)
   {
	   x+=cx;
	   y+=cy;
	   z+=cz;
   }
};
// COpenGLWnd

class COpenGLWnd : public CWnd
{
	DECLARE_DYNAMIC(COpenGLWnd)

public:
	COpenGLWnd();
	virtual ~COpenGLWnd();

/** CGLDispList
DESC:-this is an helper class which let you create "display list objects",
       use these objects to define the key elements in your scene (a disp.
	   list is faster than the corresponding GL commands).
      -Through the class members functions you have total control on a
       single display list.
      -An isolated display list save OGL parameters before execution
	   (so it's not affected by preceding transformations or settings).
*******/
	class CGLDispList
	{
	friend class COpenGLWnd;
	private:
		BOOL m_bIsolated;
		int m_glListId;
	public:
		CGLDispList();  // constructor
		~CGLDispList(); // destructor
		void StartDef(BOOL bImmediateExec=FALSE);// enclose a disp.list def.
		void EndDef();
		void Draw();// execute disp list GL commands 
		void SetIsolation(BOOL bValue) {m_bIsolated=bValue;}; // set isolation property
	};

// general drawing functions
	void DrawBox(CPoint3D org, float x, float y, float z);

/* Stock Display lists functions
DESC.: these display lists are internally organized in a vector (20 max),
       you have control on definition and redrawing only. 
       use them for background elements which are to be drawn everytime
       all together.
NOTE: between BeginStockDispList and EndStockDispList should be present 
OpenGL calls only (see documentation for which are allowed and how are them treated)
*/
	void StartStockDListDef();	// allocates a new stock display list entry and opens a display list definition
	void EndStockListDef();		// closes a stock display list definition
	void DrawStockDispLists();	// executes all the stock display lists
	void ClearStockDispLists(); // deletes all the stock display lists
// Information retrieval function
	const CString GetInformation(InfoField type);
// Mouse cursor function
	void SetMouseCursor(HCURSOR mcursor=NULL);
// Attribute retrieval function
	double GetAspectRatio() {return m_dAspectRatio;};
// Rendering Context switching
	void BeginGLCommands();// use to issue GL commands outside Overridables
	void EndGLCommands();// i.e: in menu event handlers, button events handler etc.
// font stuff
	void MakeFont();
	void PrintString(const char* str);

// Overridables
	virtual void OnCreateGL(); // override to set bg color, activate z-buffer, and other global settings
	virtual void OnSizeGL(int cx, int cy); // override to adapt the viewport to the window
	virtual void OnDrawGL(); // override to issue drawing functions
#if USE_GDI
	virtual void OnDrawGDI(CPaintDC *pDC); // override to issue GDI drawing functions
#endif
	virtual void VideoMode(ColorsNumber &c,ZAccuracy &z,BOOL &dbuf); // override to specify some video mode parameters
	void CopyToClipboard();
	void SetClearCol(COLORREF rgb);

protected:
	CDC* m_pCDC;	// WinGDI Device Context
	COLORREF m_ClearCol;	// keep local copy

private:
	HGLRC m_hRC;	// OpenGL Rendering Context
	HCURSOR m_hMouseCursor;	// mouse cursor handle for the view
	CPalette m_CurrentPalette; // palettes
	CPalette* m_pOldPalette;
	CRect m_ClientRect;    // client area size
	double m_dAspectRatio;    // aspect
	int m_DispListVector[20];	// Internal stock display list vector
	BOOL m_bInsideDispList;	// Disp List definition semaphore
	BOOL m_bExternGLCall;
	BOOL m_bExternDispListCall;
	GLuint m_FontListBase;
	BOOL m_bGotFont;
// initialization helper functions
	unsigned char ComponentFromIndex(int i, UINT nbits, UINT shift);
	void CreateRGBPalette();
	BOOL bSetupPixelFormat();

private:
	afx_msg int OnCreate(LPCREATESTRUCT lpCreateStruct);
	afx_msg void OnDestroy();
	afx_msg BOOL OnEraseBkgnd(CDC* pDC);
	afx_msg void OnSize(UINT nType, int cx, int cy);
	afx_msg BOOL OnSetCursor(CWnd* pWnd, UINT nHitTest, UINT message);
	afx_msg void OnPaint();


	DECLARE_MESSAGE_MAP()



};


