// GLSelectableScatterGraph.cpp : implementation file
//

#include "stdafx.h"
#include "KlustaWin.h"
#include "GLSelectableScatterGraph.h"
#include <float.h>


/*
This class adds user selectability to the scattergraph.
To start making a selection call StartMakeSel().
After this call the user can draw any closed shape on the screen, and 
the program can detect which data points fall within the shape.
At the moment the only use made of this is ZoomSel, which sets the 
axes so as to zoom in on the selected points.
It would be easy to modify the class to carry out other functions on the
selected points.
*/
// CGLSelectableScatterGraph

IMPLEMENT_DYNAMIC(CGLSelectableScatterGraph, CGLScatterGraph)

CGLSelectableScatterGraph::CGLSelectableScatterGraph()
{
	m_bMakeSel=FALSE;
	m_SelPts.SetSize(0,100);

//	test_num=0;
}

CGLSelectableScatterGraph::~CGLSelectableScatterGraph()
{
	m_SelPts.RemoveAll();
}


BEGIN_MESSAGE_MAP(CGLSelectableScatterGraph, CGLScatterGraph)
	ON_WM_LBUTTONDOWN()
	ON_WM_LBUTTONUP()
	ON_WM_MOUSEMOVE()
END_MESSAGE_MAP()



// CGLSelectableScatterGraph message handlers
void CGLSelectableScatterGraph::StartMakeSel()
{
	m_bMakeSel=TRUE;
	m_OldRotationType=GetRotationType();
	SetRotationType(0);
}
void CGLSelectableScatterGraph::CancelSel()
{
	m_bMakeSel=FALSE;
	m_SelPts.RemoveAll();
	SetRotationType(m_OldRotationType);
	Invalidate();
}

void CGLSelectableScatterGraph::OnLButtonDown(UINT nFlags, CPoint point) 
{
	if (m_bMakeSel)
	{
		if (m_SelPts.GetSize()!=0)
		{
			m_SelPts.RemoveAll();
//			m_SelList.RemoveAll();
			InvalidateRect(NULL);
		}
		m_SelPts.Add(point);
		m_PrevPt=point;
	}
	
	CGLScatterGraph::OnLButtonDown(nFlags, point);
}

void CGLSelectableScatterGraph::OnLButtonUp(UINT nFlags, CPoint point) 
{
	if (m_bMakeSel==TRUE && m_SelPts.GetSize()>0)
	{
#if USE_GDI
		CClientDC dc( this );
		int r=GetRValue(m_ClearCol);
		int g=GetGValue(m_ClearCol);
		int b=GetBValue(m_ClearCol);

		if (r<128 && g<128 && b<128)
			dc.SelectStockObject(WHITE_PEN);	// use white pen if dark background
		else
			dc.SelectStockObject(BLACK_PEN);	// use black pen if light background
		dc.MoveTo( m_PrevPt );
		dc.LineTo( point );
		dc.LineTo(m_SelPts.GetAt(0));
		m_SelPts.Add(point);
#endif
		m_bMakeSel=FALSE;

//		MakeSelList();
//		GetParent()->SendMessage(DONE_SELECTING);
	}
	
	CGLScatterGraph::OnLButtonUp(nFlags, point);
}

void CGLSelectableScatterGraph::OnMouseMove(UINT nFlags, CPoint point) 
{
// m_bMakeSel is set true by button click, and Capture is set
// in OnLButtonDown, which starts the drawing of the selection circle
	
	if (m_bMakeSel && m_SelPts.GetSize()>0)	// check whether had a mouse down
	{
		m_SelPts.Add(point);
#if USE_GDI
		CClientDC dc( this );

		int r=GetRValue(m_ClearCol);
		int g=GetGValue(m_ClearCol);
		int b=GetBValue(m_ClearCol);

		if (r<128 && g<128 && b<128)
			dc.SelectStockObject(WHITE_PEN);	// use white pen if dark background
		else
			dc.SelectStockObject(BLACK_PEN);	// use black pen if light background
		// Draw a line from the previous detected point in the mouse
		// drag to the current point.
		dc.MoveTo( m_PrevPt );
		dc.LineTo( point );
#else
		Invalidate();
#endif
		m_PrevPt = point;
	}
	CGLScatterGraph::OnMouseMove(nFlags, point);
}
void CGLSelectableScatterGraph::OnDrawGL()
{
	CGLScatterGraph::OnDrawGL();	// draw axes and data

	int i;
#if !USE_GDI	// use OpenGL to draw selection curve
		GLint viewport[4];
		glGetIntegerv(GL_VIEWPORT,viewport);
TRACE("viewport = %d, %d, %d, %d\n",viewport[0],viewport[1],viewport[2],viewport[3]);

		GLdouble mvmatrix[16],projmatrix[16];
		GLdouble wx,wy,wz;
		glGetDoublev(GL_MODELVIEW_MATRIX,mvmatrix);
		glGetDoublev(GL_PROJECTION_MATRIX,projmatrix);


	int r=GetRValue(m_ClearCol);
	int g=GetGValue(m_ClearCol);
	int b=GetBValue(m_ClearCol);

	if (r<128 && g<128 && b<128)	// use white pen if dark background
		glColor3f(1.f, 1.f, 1.f);
	else
		glColor3f(0.f, 0.f, 0.f);
	
	int count=m_SelPts.GetSize();
	if (count>2)
	{
		CPoint pt;
		glBegin(GL_LINE_STRIP);
		for (i=0;i<count; i++)
		{
			pt=m_SelPts.GetAt(i);
// UnProject converts actual device (screen) coords into 
// model coords. Note in OpenGL the screen coords grow upwards, 
// which is the opposite of GDI coords, 
// so need to invert GDI coords from m_SelPts.
// Rendering converts the model coords back to device coords,
// so we end up drawing the mouse track in m_SelPts in the correct
// screen locations, like a GDI call!
			gluUnProject(pt.x,viewport[3]-pt.y-1,0,
				mvmatrix,projmatrix,viewport,
				&wx,&wy,&wz);
			glVertex3f(wx,wy,wz);
		}
		pt=m_SelPts.GetAt(0);	// close the polygon
		gluUnProject(pt.x,viewport[3]-pt.y-1,0,
			mvmatrix,projmatrix,viewport,
			&wx,&wy,&wz);
		glVertex3f(wx,wy,wz);
		glEnd();
	}
#endif
}

#if USE_GDI
void CGLSelectableScatterGraph::OnDrawGDI(CPaintDC *pDC)
{
TRACE("Entering CGLSelectableScatterGraph::OnDrawGDI\n");	
	int i;

	int r=GetRValue(m_ClearCol);
	int g=GetGValue(m_ClearCol);
	int b=GetBValue(m_ClearCol);

	if (r<128 && g<128 && b<128)
		pDC->SelectStockObject(WHITE_PEN);	// use white pen if dark background
	else
		pDC->SelectStockObject(BLACK_PEN);	// use black pen if light background
	
// draw the selection circle
	int count=m_SelPts.GetSize();
	if (count>2)
	{
		pDC->MoveTo(m_SelPts.GetAt(0));
		for (i=1;i<count; i++)
			pDC->LineTo(m_SelPts.GetAt(i));
		pDC->LineTo(m_SelPts.GetAt(0));
	}
TRACE("Leaving CGLSelectableScatterGraph::OnDrawGDI\n");	
}
#endif

BOOL CGLSelectableScatterGraph::ZoomSel()
{
TRACE("entering ZoomSel\n");
#ifdef _DEBUG
	int errNum;
#endif
	int i,j,count,selPtCount;
	int id=0;
	GLint rv;
	GLfloat token;
	GLfloat *pBuf;
#ifdef _DEBUG
		int hitCount=0;
#endif


// m_SelPts is a CArray of CPoints holding the coordinates drawn previously by the user
	selPtCount=m_SelPts.GetSize();
	if (selPtCount>2)
	{
// make a region from selPts, need a standard array not a template array
		CPoint *pt=new CPoint[selPtCount];
		for (i=0; i<selPtCount; i++)
			pt[i]=m_SelPts.GetAt(i);
		CRgn rgn;
		VERIFY(rgn.CreatePolygonRgn(pt,selPtCount,ALTERNATE));
		delete [] pt;

// Get a list of coordinates using feedback mode.
// Don't know "correct" way to calculate how much memory needed for buffer.
// Experiment shows the axes drawing has 96 floats overhead,
// + 4 per point + 7 per SelPt
// give some spare, in case miscalculated !!
#if USE_GDI
		pBuf=new GLfloat[200+m_Count*4];
#else
		pBuf=new GLfloat[200+m_Count*4+selPtCount*7];
#endif

		BeginGLCommands();
#if USE_GDI
		glFeedbackBuffer(200+m_Count*4,GL_3D,pBuf);
#else
		glFeedbackBuffer(200+m_Count*4+selPtCount*7,GL_3D,pBuf);
#endif
		glRenderMode(GL_FEEDBACK);
#ifdef _DEBUG
		errNum=glGetError();
		TRACE("error on setting render mode = %s\n",gluErrorString(errNum));
		{
			int rvRenderMode;
			glGetIntegerv(GL_RENDER_MODE,&rvRenderMode);
			CString mode;
			if (rvRenderMode==GL_RENDER)
				mode="Render";
			else if (rvRenderMode==GL_FEEDBACK)
				mode="Feedback";
			else
				mode="Unknown";
		TRACE("Render mode = %s\n",mode);
		}
#endif
		EndGLCommands();
TRACE("ZoomSel just about to update window to store coords\n");

// Cause the window to be redrawn, thus calling the GL drawing code in feedback mode.
// This puts a list of window coords for each item drawn into pBuf, along with a token specifying type.
// The window coords are GDI-type, but increasing Y upwards

		Invalidate();
		UpdateWindow();

		BeginGLCommands();
		rv=glRenderMode(GL_RENDER);
TRACE("return val from setting RenderMode = %d\n",rv);

//////////
// Now find which data points have which screen coords.
		int *pInSel=new int[m_Count];	// for each pt, will be 1 if in, 0 if out

// Need to know the viewport, so can invert Y values.
// (OpenGL Y coords increase upwards, Win coords increase downwards.)
		GLint viewport[4];
		glGetIntegerv(GL_VIEWPORT,viewport);
		EndGLCommands();

// Now find which data points have which screen coords.
		count=rv;
		while (count)
		{
			token=pBuf[rv-count];
			count--;
// Ignore all drawings stuff except points
			if (token==GL_POINT_TOKEN)
			{
//				TRACE("Point token\n");
				GLdouble coords[3];
				for (j=0; j<3; j++)
				{
//					TRACE("%4.2f ",pBuf[rv-count]);
					coords[j]=pBuf[rv-count];
					count--;
				}
//				TRACE("\n");
				CPoint pt;
				pt.x=int(coords[0]);
				pt.y=int(viewport[3]-coords[1]-1);
				pInSel[id++]=rgn.PtInRegion(pt);
#ifdef _DEBUG
				TRACE("pt %d has hit=%d\n",id-1,pInSel[id-1]);
				if (rgn.PtInRegion(pt))
					hitCount++;
#endif
			}
		}

// Now we set the axis scales to draw the selected points,
// (plus any others within the axes limits).
		m_bAutoScaleX=m_bAutoScaleY=m_bAutoScaleZ=FALSE;

// find max & min
		float xMax,yMax,zMax,xMin,yMin,zMin;
		xMax=yMax=zMax=-FLT_MAX;
		xMin=yMin=zMin=FLT_MAX;

// drawCount is index of points which are drawn within old axes, 
// and therefore appear in pSelList.
		int drawCount=-1;	
		for (i=0; i<m_Count; i++)
		{
// Determine whether point was being drawn (is within old axes)
// and so will be in selection list (as either 1 if within shape or 0 if outside it).
			if (!PtWithinAxes(m_pDat[i*3],m_pDat[i*3+1],m_pDat[i*3+2]))
				continue;
			drawCount++;
			if (pInSel[drawCount]==0)
				continue;
			xMax=max(m_pDat[i*3],xMax);
			xMin=min(m_pDat[i*3],xMin);
			yMax=max(m_pDat[i*3+1],yMax);
			yMin=min(m_pDat[i*3+1],yMin);
			zMax=max(m_pDat[i*3+2],zMax);
			zMin=min(m_pDat[i*3+2],zMin);
		}

// Prettify axis scales.
		m_MaxX=NextAbove(xMax,5);
		m_MinX=NextBelow(xMin,5);
		m_MaxY=NextAbove(yMax,5);
		m_MinY=NextBelow(yMin,5);
		m_MaxZ=NextAbove(zMax,5);
		m_MinZ=NextBelow(zMin,5);
		Invalidate();

		delete [] pBuf;
		delete [] pInSel;
		rgn.DeleteObject();
	}
	CancelSel();

#ifdef _DEBUG
TRACE("leaving ZoomSel, %d points on screen, %d in selection\n",id,hitCount);
#endif

	return TRUE;
}


