	// add a line of code into the chart library to prevent Xlabel to be rotated
	Chart.Scale = Chart.Scale.extend({
		oldCalculateXLabelRotation :	Chart.Scale.prototype.calculateXLabelRotation,
		calculateXLabelRotation: function() {
			var end_point = this.endPoint;
			this.oldCalculateXLabelRotation();
			this.xLabelRotation = 0;
			this.endPoint = end_point;
		}
	});
	
	/**
	 * Plot function graph
	 * @param {element} ele - the element to be needed to draw the graph [ ie. $('#Stage_graph') ]
	 * @param {Array} eqns - array of functions f(x) used to calculate value of Y
	 * @param {Object} xAxisConfig - configuration for x axis :
	 * 							{number} minX - starting value of x-axis
	 * 							{number} maxX - maximum x value to be plot on graph
	 * 							{number} step - increment x value for each point
	 * 							{number} showXLabels - number of x labels shown on the graph
	 * 							{number} labelDecimal - number of decimal to be displayed for x labels (default: 0)
	 * @param {Array} dataStruc - array of Object containing data structure of each dataset (line color. See Chart.js documentation: Line Chart-data structure) **may not needed
	 * @param {Object} config - contain configuration such as label size, Y axis scale. See Chart.js documentation: Getting started-Global chart configuration and Line Chart-Chart options **may not needed
	 **/
	function FunctionGraph(ele, eqns, xAxisConfig, dataStruc, config) {
		// default values for graph
		var defaultXAxis = {
			minX: 0,
			maxX: 10,
			step: 1,
			showXLabels: 11,
			labelDecimal: 0
		}
		var defaultConfig = {
			scaleShowGridLines : false,
			animation: false,
			animationSteps: 1,
			responsive: false,
			showTooltips: false,
			scaleLineColor: "rgba(0,0,0,1.0)",
			scaleFontColor: '#000000',
			datasetStrokeWidth: 1,
			pointDot: false,
			pointDotRadius: 3,
			bezierCurve: false,
			datasetFill: false,
			scaleBeginAtZero: false
		};
		
		this.xAxisConfig = defaultXAxis;
		$.extend(this.xAxisConfig, xAxisConfig);
		this.config = defaultConfig;
		if (typeof config !== 'undefined') {
			$.extend(this.config, config);
		}
		this.init(ele, eqns, this.xAxisConfig, dataStruc, this.config);
	}
	FunctionGraph.prototype.init = function(ele, eqns, xAxisConfig, dataStruc, config) {
		var minX = xAxisConfig.minX;
		var maxX = xAxisConfig.maxX;
		var step = xAxisConfig.step;
		var label_num = xAxisConfig.showXLabels;
		var labelDecimal = xAxisConfig.labelDecimal;

		// dataset default value
		var defaultDataStruc = {
			label: "dataset",
			fillColor : "rgba(0,0,220,0.2)",
			strokeColor : "rgba(0,0,225,1)",
			pointColor : "rgba(0,0,220,1)",
			pointStrokeColor : "#fff",
			pointHighlightFill : "#fff",
			pointHighlightStroke : "rgba(0,0,220,1)"
		};
		// change element to canvas
		var thiscanvasstyle = ele.attr('style');  // unused, might be useful if style changed unexpected
		var ele_id = ele.attr('id'); // remember the id for later use
		if (ele.prop('tagName').toLowerCase() != 'canvas' ) {
			ele.replaceWith('<canvas id="' + ele.attr('id') + '" class="' + ele.attr('class') + '" style="' + ele.attr('style') + '">' + ele.html() + '</canvas>');
		}
		// calculate label x axis
		var total_point = (maxX - minX) / step + 1;
		var label_index = Math.floor((total_point-1)/(label_num-1));
		var labels = [];
		for (var i = 0; i < total_point; i++) {
			var x = minX + i * step;
			var label = !(i%label_index)? x.toFixed(labelDecimal):'';
			labels.push(label);
		}

		// calculate value for all equation
		var datasets = [];
		var eval = function(eqn, index, array) {
			var vals = [];
			for (var i = 0; i < total_point; i++) {
				var x = minX + i * step;
				vals.push(eqn(x));
			}
			var dataset = $.extend(true, [], defaultDataStruc);
			// if new data structures are given 
			if (typeof dataStruc !== 'undefined') {
				dataset = $.extend({}, dataset, dataStruc[index]);
			}
			dataset.data = vals;
			datasets.push(dataset);
		}
		eqns.forEach(eval);

		// create graph
		var lineChartData = {
			labels : labels,
			datasets : datasets
		}

		ele = $('#'+ele_id)[0];
		var ctx = ele.getContext("2d");

		this.graph = new Chart(ctx).Line(lineChartData, config);
	}
	FunctionGraph.prototype.destroy = function() {
		this.graph.destroy();
	}
	
	/**
	 * Plot Data graph
	 * 
	 **/
	function DataGraph(ele, eqns, xAxisConfig, dataStruc, customConfig) {
		// default values for graph
		var defaultConfig = {
			scaleShowGridLines : false,
			animation: false,
			animationSteps: 1,
			responsive: false,
			showTooltips: false,
			scaleLineColor: "rgba(0,0,0,1.0)",
			scaleFontColor: '#000000',
			datasetStrokeWidth: 1,
			pointDot: false,
			pointDotRadius: 3,
			bezierCurve: false,
			datasetFill: false,
			scaleBeginAtZero: false
		};
		var maxX = xAxisConfig.maxX;
		var label_num = xAxisConfig.showXLabels;
		this.total_point = maxX + 1;
		this.label_num = Math.floor(maxX / (label_num-1));
		this.currentMeasurement = 0;
		this.eqns = eqns;
		this.config = defaultConfig;
		if (typeof customConfig !== 'undefined') {
			$.extend(this.config, customConfig);
		}

		this.init(ele, eqns, dataStruc);
	};
	DataGraph.prototype.init = function(ele, eqns, dataStruc) {
		// change element to canvas
		var thiscanvasstyle = ele.attr('style');  // unused, might be useful if style changed unexpected
		var ele_id = ele.attr('id'); // remember the id for later use
		if (ele.prop('tagName').toLowerCase() != 'canvas' ) {
			ele.replaceWith('<canvas id="' + ele.attr('id') + '" class="' + ele.attr('class') + '" style="' + ele.attr('style') + '">' + ele.html() + '</canvas>');
		}
		
		//creating label 
		var labelx = [];
		var dataPoints = [];
		for (var i = 0; i < this.total_point; i++) {
			labelx.push(i%this.label_num==0?i.toString():'');
		}
		
		// default dataset value
		var defaultDataStruc = {
			label: "dataset",
			fillColor : "rgba(0,0,220,0.2)",
			strokeColor : "rgba(0,0,225,1)",
			pointColor : "rgba(0,0,220,1)",
			pointStrokeColor : "#fff",
			pointHighlightFill : "#fff",
			pointHighlightStroke : "rgba(0,0,220,1)"
		};
		
		// initialize graph
		var datasets = [];
		var eval = function(eqn, index, array) {
			var dataset = $.extend(true, [], defaultDataStruc);
			// if new data structures are given 
			if (typeof dataStruc !== 'undefined') {
				dataset = $.extend({}, dataset, dataStruc[index]);
			}
			dataset.data = [eqn(0)];
			datasets.push(dataset);
		}
		eqns.forEach(eval);
		
		var lineChartData = {
			labels : labelx,
			datasets : datasets
		}

		ele = $('#'+ele_id)[0];
		var ctx = ele.getContext("2d");
		this.graph = new Chart(ctx).Line(lineChartData, this.config);
	}
	DataGraph.prototype.destroy = function() {
		this.graph.destroy();
	}
	DataGraph.prototype.addPoint = function() {
		this.currentMeasurement++;
		// calculate value for each eqn
		var currentMeasurement = this.currentMeasurement;
		var graph = this.graph;
		var vals = [];
		this.eqns.forEach(function(eqn, index, array) {
			vals.push(eqn(currentMeasurement));
		});

		if (this.currentMeasurement < this.total_point) {
			// if the there is still have space for new point
			// add point into the graph
			this.eqns.forEach(function(eqn, index, array) {
				var point_data = {
					value : vals[index],
					label : currentMeasurement.toString(),
					datasetLabel: graph.datasets[index].label,
					x: graph.scale.calculateX(currentMeasurement), // this properties need modified if needed
					y: graph.scale.endPoint,
					strokeColor : graph.datasets[index].pointStrokeColor,
					fillColor : graph.datasets[index].pointColor
				}
				graph.datasets[index].points.push(new graph.PointClass(point_data));
			});
			graph.update();
		} else {
			// remove last data and add new data
			graph.removeData();
			var label = currentMeasurement%10==0?currentMeasurement.toString():'';
			graph.addData(vals,label);
		}
		return vals;
	}
	DataGraph.prototype.addPoints = function(num_point) {
		var result = [];
		var graph = this.graph;
		// just add point there is space for more point
		while (this.currentMeasurement < this.total_point && num_point > 0) {
			this.currentMeasurement++;
			// calculate value for each points
			var vals = [];
			var currentMeasurement = this.currentMeasurement;
			this.eqns.forEach(function(eqn, index, array) {
				vals.push(eqn(currentMeasurement));
			});
			// add point into the graph
			this.eqns.forEach(function(eqn, index, array) {
				var point_data = {
					value : vals[index],
					label : currentMeasurement.toString(),
					datasetLabel: graph.datasets[index].label,
					x: graph.scale.calculateX(currentMeasurement), // this properties need modified if needed
					y: graph.scale.endPoint,
					strokeColor : graph.datasets[index].pointStrokeColor,
					fillColor : graph.datasets[index].pointColor
				}
				graph.datasets[index].points.push(new graph.PointClass(point_data));
			});
			num_point--;
			result.push(vals);
		}
		
		if (num_point == 0) {
			graph.update();
			return result;
		}
		
		var number_to_keep = this.total_point - num_point;
		var points;
		// remove previous unwanted labels
		graph.scale.xLabels = graph.scale.xLabels.slice(-number_to_keep);
		graph.scale.valuesCount -= num_point;
		graph.scale.fit();
		
		// remove unwanted points
		for (var i = 0; i < graph.datasets.length; i++) {
			graph.datasets[i].points = graph.datasets[i].points.slice(-number_to_keep);
		}
	
		for (var i=0; i<num_point; i++) {
			var currentMeasurement = ++this.currentMeasurement;
			var label = currentMeasurement%this.label_num==0?currentMeasurement.toString():'';
			// add labels
			graph.scale.addXLabel(label);
			// add points
			var graph = this.graph;
			var vals = [];
			this.eqns.forEach(function(eqn, index, array) {
				var val = eqn(currentMeasurement)
				var point_data = {
					value : val,
					label : currentMeasurement.toString(),
					datasetLabel: graph.datasets[index].label,
					x: graph.scale.calculateX(graph.scale.valuesCount+1),
					y: graph.scale.endPoint,
					strokeColor : graph.datasets[index].pointStrokeColor,
					fillColor : graph.datasets[index].pointColor
				}
				graph.datasets[index].points.push(new graph.PointClass(point_data));
				vals.push(val);
			});
			result.push(vals);
		}
		graph.update();
		return result;
	}